/**************************************************************************
 *
 * Beispiel, bei dem der Eltern-Prozess den Exit-Status eines Kind-
 * Prozesses auswertet
 *                                                                        
 * Ersteller :       Ingo Phleps                                          
 *                                                                        
 * Stand :           $Date: 2005/12/03 $
 *                                                                        
 * Zielsystem:       UNIX                                                 
 *                                                                        
 * bersetzen:       cc exitstatus.c -o exitstatus
 *
 * Aufruf:	     ./exitstatus
 *
 * Dieses Programm zeigt die bergabe des Exit-Status vom Kind- an den Eltern-
 * Prozess und dessen Auswertung im Eltern-Prozess:
 *
 * Der Eltern-Prozess startet einen Kind-Prozess, der sich selbst 
 * mit Exit-Status 5 beendet.
 * Nach dem Start des Kind-Prozesses wartet der Eltern-Prozess auf das Ende
 * des Kind-Prozesses, wertet dessen Exit-Status aus und zeigt ihn an. 
 * 
 * Funktionen des Moduls:
 *
 *   main		()   Hauptprogramm
 *                                                                        
 **************************************************************************/

# include <stdio.h>
# include <time.h>
# include <unistd.h>
# include <stdlib.h>
# include <sys/types.h>
# include <sys/wait.h>


	/* --- Konstanten ----------------------------------------------- */

	/* Exit-Status, mit dem sich der Kind-Prozess selbst beendet	  */
# define CHILD_EXIT_STATUS  5


	/* --- Typdeklarationen ----------------------------------------- */



	/* --- globale Variablen ---------------------------------------- */

static char                  /* RCS-Versionsnummer ...                    */
  * header = "$Id: exitstatus.c $";



	/* --- Prototypen ----------------------------------------------- */

int main ( int argc, char * argv [] );



	/* --- Funktionen ----------------------------------------------- */

int main ( int argc, char * argv[] )

/* Hauptprogramm
 *
 * Argumente:
 *   keine
 *
 * Ergebnis:
 *   EXIT_SUCCESS
 *
 * Das Programm startet einen Kind-Prozess, der nach einer kurzen Wartezeit
 * sein Ende ankuendigt und sich dann selbst mit Exit-Status CHILD_EXIT_STATUS
 * beendet.
 *
 * Der Eltern-Prozess wartet nach dem Start des Kind-Prozesses auf dessen
 * Ende, liest dessen Exit-Status und gibt eine entsprechende Meldung aus.
 */
{
  pid_t
    childPid;		     /* Prozess-ID des Kind-Prozesses		  */

  int
    waitResult,		     /* Funktionsergebnis von wait ()		  */
    childStatus;	     /* mit wait() gelesener Exit-Status	  */

    
  printf ( "Dies ist der Eltern-Prozess mit Prozess-ID %5d\n",
	                                                  (int) getpid () );

    
	  /* Kind-Prozess starten					  */
  switch ( childPid = fork () )
  {
    case -1:		     /* Fehler beim starten des Kind-Prozesses:	  */
		/* Fehlermeldung ausgeben				  */
	perror ( "fork()" );

	break;

    case 0:		     /* Kind-Prozess:				  */
		/* kurze Zeit warten					  */
	sleep ( 5 );

		/* Ende des Kind-Prozesses und Exit-Status anzeigen	  */
	printf ( "\nKind-Prozess mit Prozess-ID %5d beendet sich"
		                                 " mit Exit-Status = %d\n",
		 (int) getpid (), CHILD_EXIT_STATUS );

		/* Kind-Prozess mit vorgegebenem Exit-Status beenden	  */
	exit ( CHILD_EXIT_STATUS ) ;
	
		/* NOTREACHED */
	break;  /* Nur der Ordnung halber. Sicher ist sicher.		  */

    default:		     /* Eltern-Prozess:				  */
		/* Noch keine Aktion					  */
	break;
  }

  printf ( "\nEltern-Prozess mit PID %5d wartet auf das Ende des"
	                                               " Kind-Prozesses\n",
	   (int) getpid () );


	  /* Ende des Kind-Prozesses abwarten und dessen Exit-Status
	   * lesen.							  */
  waitResult = wait ( & childStatus );

	  /* Fehler beim lesen des Exit-Status?				  */
  if ( waitResult < 0 ) 
  {
	    /* Fehlermeldung ausgeben					  */
    perror ( "Fehler bei wait()" );
  }
	  /* Regulres Ende des Kind-Prozesses?				  */
  else if ( WIFEXITED ( childStatus ) )
  {
	      /* Prozess-ID des Kindprozesses und dessen Exit-Status
	       * anzeigen						  */
    printf ( "\nEltern-Prozess mit PID %d hat Exit-Status %d von "
	                      "Kind-Prozess mit \nProzess-ID %d gelesen.\n\n",
	     (int) getpid(), WEXITSTATUS ( childStatus ), (int) waitResult );
  }
  else
  {
	    /* Hinweis auf Abbruch des Prozesses ausgeben.
	     *
	     * Anmerkung: Mit WIFSIGNALED() und WTERMSIG()
	     *	    bzw. WIFSTOPPED() und WSTOPSIG() knnte die
	     *            Ursache bei Bedarf genauer analysiert werden  */
    printf ( "\nKind-Prozess mit Prozess-ID %d wurde abgebrochen\n\n",
	                  (int) waitResult );
  }

  return EXIT_SUCCESS;	     /* Returncode: OK				  */
}

