/**************************************************************************
 * $Header: //depot/rech_org/skript/src/pipe.c#2 $
 *
 * Beispiel zu einer Pipe
 *
 * erstellt von:	Ingo Phleps
 *
 * Stand:		$Date: 2004/01/28 $
 *
 * Beispiel zur Vorlesung "Rechnerorganisation"     6. Halbjahr
 * Fachrichtung Nachrichtentechnik an der Berufsakademie Stuttgart
 * 
 * Dieses Programm soll das Anlegen einer Pipe zu einem Kindproze und
 * den Transport von Daten durch diese Pipe zeigen.
 *
 *
 * Funktionen:
 *   main		()   Hauptprogramm
 *   WritePipeData	()   Daten als Strings ber Pipe verschicken
 *   ReadPipeData	()   Daten aus der Pipe lesen u. gelesene Daten
 *			     als String anzeigen
 *   GetTimeStr		()   Aktuelle Uhrzeit als String liefern
 *
 **************************************************************************/

/* ---- Definitionen zum ein-/ausschalten bedingter Uebersetzung -------- */

# define _POSIX_SOURCE    1  /* Uebersetzen entsprechend POSIX Standard	  */


/* ---- allgemeine Include-Dateien -------------------------------------- */

# include <stdio.h>
# include <unistd.h>
# include <stdlib.h>
# include <time.h>
# include <sys/types.h>
# include <sys/wait.h>


/* ---- projektspezifische Include-Dateien ------------------------------ */



/* ---- Symbolische Konstanten, Makros: # define ... -------------------- */



/* ---- Aufzaehlungstypen: enum ... ------------------------------------- */



/* ---- Typdeklarationen von Strukturen und unions: --------------------- */

  

/* ---- Definitionen von globalen und static-Variablen: ----------------- */

# ifndef __LINT__
	/* Objekt-Datei und Programm mit RCS-Infostrings kennzeichnen	  */
volatile static char
  * rcsHeader = "@(#)$Header: //depot/rech_org/skript/src/pipe.c#2 $" ;
# endif   /* __LINT__  */


/*  */
/* ---- Funktionsprototypen von static-Funktionen: ---------------------- */

	/* Hauptprogramm						  */
int main           ( void );

	/* Daten als Strings ber Pipe verschicken			  */
void WritePipeData ( int fdPipeWr );

	/* Daten aus der Pipe lesen u. gelesene Daten als String anzeigen */
void ReadPipeData  ( int fdPipeRd );

	/* Aktuelle Uhrzeit als String liefern				  */
char * GetTimeStr  ( void );



/* ---- Funktionen: ----------------------------------------------------- */

int main ( void )

/* Funktionsergebnis:
 *   EXIT_SUCCESS	Kein Fehler aufgetreten
 *   EXIT_FAILURE	Programm wegen Fehler abgebrochen
 *
 * Funktionsparameter:
 *   keine
 *
 * Globale Variablen:
 *   keine
 *
 * Hauptprogramm
 *
 * Das Hauptprogramm legt eine Pipe an und spaltet danach einen Kindproze
 * ab.
 * Der Kindproze schliet das Lese-Ende der Pipe und sendet mehrere
 * Nachrichten durch die Pipe.
 * Der Elternproze schliet das Schreib-Ende der Pipe und wartet auf Nach-
 * richten vom Kindproze. Alle empfangenen Nachrichten werden nach stdout
 * ausgegeben.
 */
/*  */
{
  pid_t
    childPid;		     /* Proze-ID des Kindprozesses		  */

  int
    msgPipe [2];	     /* Pipe vom Kind- zum Elternproze:
			      *   msgPipe [0]: Lese-Ende der Pipe
			      *   msgPipe [1]: Schreib-Ende der Pipe	  */


	  /* Pipe anlegen:
	   * Fehler ?							  */
  if ( 0 != pipe ( msgPipe ) )
  {
	    /* Fehlermeldung ausgeben, Programm abbrechen		  */
    perror ( "pipe()" );

    exit ( EXIT_FAILURE );
  }

	  /* Alle Zwischenpuffer des Betriebssystems leeren.
	   * Wre in diesem Fall zwar nicht notwendig, aber sicher ist
	   * sicher ...							  */
  fflush ( NULL );
  
	  /* Kindproze abspalten:					  */
  switch ( childPid = fork () )
  {
    case -1:		     /* Fehler beim starten des Kindprozesses:	  */
	perror ( "fork()" ); /* Fehlermeldung ausgeben			  */
	break;

    case 0:		     /* Kind-Proze:				  */
		/* Lese-Ende der Pipe schlieen				  */
	close ( msgPipe [0] );
	msgPipe [0] = -1;    /* Deskriptor ist nicht mehr ungltig	  */

		/* Daten ber Pipe verschicken				  */
	WritePipeData ( msgPipe [1] );
	
		/* Aufrumen: Schreib-Ende der Pipe schlieen		  */
	close ( msgPipe [1] );
	msgPipe [1] = -1;    /* Deskriptor ist nicht mehr ungltig	  */
	
		/* Kindproze beenden					  */
	exit ( EXIT_SUCCESS );
		/* NOTREACHED */
	break;	/* Nur der Ordnung halber. Sicher ist sicher.		  */

    default:		     /* Eltern-Proze:				  */
		/* Schreib-Ende der Pipe schlieen			  */
	close ( msgPipe [1] );
	msgPipe [1] = -1;    /* Deskriptor ist nicht mehr ungltig	  */

		/* Daten aus der Pipe lesen und gelesene Daten anzeigen	  */
	ReadPipeData ( msgPipe [0] );

		/* Aufrumen: Lese-Ende der Pipe schlieen		  */
	close ( msgPipe [0] );
	msgPipe [0] = -1;    /* Deskriptor ist nicht mehr ungltig	  */

	break;
	
  }  /*  Ende switch ( fork () ) */

	  /*  */
  
  wait ( NULL );	     /* Zombie-Proze vermeiden			  */

  exit ( EXIT_SUCCESS );     /* Elternproze beenden			  */

} /* Ende  main ()  */



void WritePipeData ( int fdPipeWr )

/* Funktionsergebnis:
 *   keines
 *
 * Funktionsparameter:
 *   fdPipeWr	in	File Deskriptor zum Schreiben in die Pipe
 *
 * Globale Variablen:
 *   keine
 *
 * Daten als Strings ber Pipe verschicken
 *
 * Die Funktion versendet einige Strings im Abstand von etwa 5 Sekunden 
 * ber die Pipe. 
 */
{
  int
    retStatus   =   -1,	     /* Ergebnis aufgerufener Funktionen	  */
    msgCount,		     /* Meldungszhler				  */
    msgLen;		     /* Lnge der Nachricht einschlielich ab-
			      * schlieendem '\0'			  */

  char
    sendBuffer [ 100 ];	     /* Speicher fr zu sendende Daten		  */


	  /* einige Nachrichten im Abstand einer Sekunde versenden	  */
  for ( msgCount = 1 ; msgCount <= 5 ; msgCount ++ )
  {
	    /* Nachricht zusammenstellen und Lnge einschlielich
	     * abschlieendem '\0' ermitteln				  */
    sprintf ( sendBuffer,
	      "%2d. gesendete Nachricht: Es ist jetzt %s Uhr ",
	      msgCount, GetTimeStr () );
    
    msgLen = strlen ( sendBuffer );

	    /* Nachricht ber Pipe versenden				  */
    retStatus   = write ( fdPipeWr, sendBuffer, msgLen );

	    /* Fehler ?							  */
    if ( retStatus != msgLen )
    {
      perror ( "write()" );
    }

	    /* etwa 5 Sekunden warten					  */
    sleep ( 5 );
  }
  
} /* Ende  WritePipeData ()  */


/*  */

void ReadPipeData ( int fdPipeRd )

/* Funktionsergebnis:
 *   keines
 *
 * Funktionsparameter:
 *   fdPipeRd	in	File Deskriptor zum Lesen aus der Pipe
 *
 * Globale Variablen:
 *   keine
 *
 * Daten aus der Pipe lesen und gelesene Daten als String anzeigen
 *
 * Die Funktion liest Daten aus der Pipe und gibt die empfangenen Daten als
 * Strings auf den Bildschirm aus.
 * Das Lesen von Daten wird beendet, wenn der schreibende Proze sein Ende
 * der Pipe schliet.
 */
{
  int
    readLen,		     /* Zahl der gelesenen Zeichen		  */
    msgCount;		     /* Meldungszhler				  */

  char
    recvBuffer [ 30 ];	     /* Speicher fr empfangene Daten
			      * Zu Demonstrationszwecken ist der Empfangs-
			      * puffer absichtlich kleiner als der von
			      * WritePipeData() verwendete Sendespeicher. */

  
	  /* So lange aus der Pipe lesen und gelesenen Inhalt ausgeben,
	   * bis der schreibende Proze sein Ende der Pipe schliet.
	   * read() liefert dann das Ergebnis 0				  */
  msgCount = 0;

  do
  {
	    /* Nachrichtenzhler weiterzhlen				  */
    msgCount ++ ;
    
	    /* Daten aus der Pipe lesen					  */
    readLen = read ( fdPipeRd, recvBuffer, sizeof ( recvBuffer ) -1 );

	    /* Fehler ?						  */
    if ( readLen < 0 )
    {
	      /* Fehler anzeigen					  */
      perror ( "read()" );      
    }
    else
    {
	      /* gelesene Daten als String abschlieen und ausgeben	  */
      recvBuffer [ readLen ] = '\0';
      printf ( "%s: %2d. Nachricht empfangen, %2d Zeichen: \"%s\"\n",
	       GetTimeStr(), msgCount, readLen, recvBuffer );
    }
	  
  } while ( 0 != readLen );

} /* Ende  ReadPipeData ()  */


/*  */

char * GetTimeStr ( void )

/* Funktionsergebnis:
 *   Zeiger auf String mit aktueller Uhrzeit
 *
 *	!!! ACHTUNG !!!
 *	Der als Ergebnis geliefete Zeiger zeigt auf eine static-Variable!!
 *	Deren Inhalt ndert sich beim nchsten Funktionsaufruf! 
 *
 * Funktionsparameter:
 *   keine
 *
 * Globale Variablen:
 *   keine
 *
 * Aktuelle Uhrzeit als String liefern
 *
 * Diese Funktion liefert als Ergebnis einen Zeiger auf eine Static-Variable,
 * die die aktuelle Zeit im Format  hh:mm:ss  enthlt. der Inhalt der
 * Variablen wird bei jedem Aufruf dieser Funktion aktualisiert. 
 */
{
  time_t
    now;		     /* aktuelle Zeit				  */
  
  static char
    timeStr [ 10 ];	     /* aktuelle Zeit als String, Format hh:mm:ss */

  
	  /* aktuelle Zeit ermitteln					  */
  time ( & now );

	  /* Zeit als String ablegen					  */
  strftime ( timeStr, sizeof ( timeStr ), "%H:%M:%S", localtime ( & now ) );

	  /* Ergebnis: Zeiger auf String mit aktueller Zeit		  */
    
  return timeStr;
	
} /* Ende  GetTimeStr ()  */

