/**************************************************************************
 *
 * Beispiel zu Threads
 *                                                                        
 * Ersteller :       Ingo Phleps                                          
 *                                                                        
 * Stand :           28.10.2000
 *                                                                        
 * Zielsystem:       UNIX                                                 
 *                                                                        
 * bersetzen:       cc pthread.c -lpthread -o pthread
 *
 * Dieses Programm soll das Verhalten von Threads sowie deren privaten und
 * gemeinsamen Speicherbereiche zeigen. Dazu schreibt jeder Thread
 * Informationen auf den Bildschirm.
 *                                                                        
 * Beispiel fr das Ergebnis des Programmes auf dem Bildschirm:          
 *
 *  Programm gestartet 
 *                    gvCallCnt           mainCounter          localCounter
 *  A  Thread 1:    * 0x804fe00 =  0    * 0xbffff7a4 =  0    * 0xbffff784 =  0
 *  B  Thread 1:    * 0x804fe00 =  1    * 0xbffff7a4 =  2    * 0xbffff784 = 15
 *  C  Thread 1:    * 0x804fe00 =  1    * 0xbffff7a4 =  2    * 0xbffff784 = 15
 *  A  Thread 1:    * 0x804fe00 =  1    * 0xbffff7a4 =  3    * 0xbffff784 =  3
 *  B  Thread 1:    * 0x804fe00 =  2    * 0xbffff7a4 =  5    * 0xbffff784 = 18
 *  A  Thread 2:    * 0x804fe00 =  2    * 0xbffff7a4 =  5    * 0xbf9ffd80 =  5
 *  B  Thread 2:    * 0x804fe00 =  3    * 0xbffff7a4 =  7    * 0xbf9ffd80 = 20
 *  C  Thread 1:    * 0x804fe00 =  3    * 0xbffff7a4 =  7    * 0xbffff784 = 18
 *  C  Thread 2:    * 0x804fe00 =  3    * 0xbffff7a4 =  7    * 0xbf9ffd80 = 20
 *  D  Ende:        * 0x804fe00 =  3    * 0xbffff7a4 =  7      (NULL) 
 *
 *									  
 * Funktionen des Moduls:
 *
 *   main		()   Hauptprogramm
 *   ModifyThreadData	()   Zhlerstnde verndern und anzeigen
 *   PrintThreadStatus	()   Zhleradressen und -inhalte anzeigen 
 *                                                                        
 **************************************************************************/

# include <stdio.h>
# include <unistd.h>
# include <stdlib.h>
# include <pthread.h>


	/* --- Typdeklarationen ----------------------------------------- */

struct lt_funcArg	     /* Daten zur bergabe an ModifyThreadData(): */
{
  char * threadName;	     /* Bezeichnung zur Unterscheidung d. Threads */

  int  * mainCntPtr;	     /* Zeiger auf mainCounter in main ()	  */
};


	/* --- globale Variablen ---------------------------------------- */

static char                  /* RCS-Versionsnummer ...                    */
  * header = "$Id: pthread.c,v 1.1 2000/10/28 16:55:07 phleps Exp $";

static int
  gvCallCnt;		     /* Zahl der Aufrufe von ModifyThreadData()	  */
  

	/* --- Prototypen ----------------------------------------------- */

int main                ( void );

void * ModifyThreadData ( struct lt_funcArg * funcArgPtr );

void PrintThreadStatus  ( char label, char * threadName,
			  int * mainCntPtr, int * localCntPtr );


	/* --- Funktionen ----------------------------------------------- */
int main ( void )

/* Hauptprogramm
 *
 * Das Hauptprogramm ruft die Funktion ModifyThreadData() sowohl direkt,
 * als auch in einem separaten Thread auf. Beide Threads schreiben
 * jeweils Daten zu ihrem Thread auf den Bildschirm.
 */
{
  pthread_t
    thread;		     /* ID des erzeugten Threads		  */

  int
    retStatus,		     /* Ergebnis aufgerufener Funktionen	  */
    mainCounter;	     /* Zhler auf dem Stack des Hauptprogramms	  */

  struct lt_funcArg	     /* Argument fr ModifyThreadData():	  */
    funcArg1,		     /*   Thread 1				  */
    funcArg2;		     /*   Thread 2				  */

    
  printf ( "Programm gestartet \n" );

	  /* Argumente fr ModifyThreadData():
	   * Die Strukturen fr beide Threads werden mit unterschiedlichen
	   * namen, aber mit dem gleichen Zeiger auf die Variable
	   * 'mainCounter' besetzt.					  */
	  /*   Thread 1:						  */
  funcArg1.threadName = "Thread 1:";
  funcArg1.mainCntPtr = & mainCounter;

	  /*   Thread 2:						  */
  funcArg2.threadName = "Thread 2:";
  funcArg2.mainCntPtr = & mainCounter;

  
  gvCallCnt   = 0;	     /* Zhler zurcksetzen			  */
  mainCounter = 0;

	  /* berschriften zu Ausgaben von PrintThreadStatus() ausgeben	  */
  printf ( "                  gvCallCnt           mainCounter          "
	   						    "localCounter\n" );
  
	  /* Funktion direkt aufrufen. Dieser Thread sei Thread 1.	  */
  ModifyThreadData ( & funcArg1 );


	  /* Funktion als separaten "Thread 2" starten. 
	   *
	   * Da das Argument fr ModifyThreadData() im Funktionsprototyp
	   * als (struct lt_funcArg *) definiert ist, pthread_create()
	   * aber eine Funktion mit Argumenttyp (void *) erwartet, wrde
	   * der Compiler bei hohem Warning-Level eine Warnung ausgeben.
	   * Durch den Type-Cast vor ModifyThreadData wird diese Warnung
	   * unterdrckt.						  */
  retStatus = pthread_create ( & thread, NULL,
			       (void *(*)(void *)) ModifyThreadData,
			       & funcArg2 );

	  /* Fehler beim starten des neuen Threads?			  */
  if ( 0 != retStatus )
  {
	    /* Fehlermeldung ausgeben					  */
    perror ( "pthread_create()" );				  
  }

  mainCounter ++ ;	/* Inhalt des Zhlers verndern			  */

	  /* Funktion nochmals direkt aufrufen				  */
  ModifyThreadData ( & funcArg1 );

	  /* warten, bis der zweite Thread fertig ist			  */
  retStatus = pthread_join ( thread, NULL );

	  /* Fehler beim warten auf das Ende von Thread 2?		  */
  if ( 0 != retStatus )
  {
	    /* Fehlermeldung ausgeben					  */
    perror ( "pthread_join()" );
  }

	  /* Label D: Zhlerdaten nochmals anzeigen			  */
  PrintThreadStatus ( 'D', "Ende:", & mainCounter, NULL );

  return EXIT_SUCCESS;	     /* Returncode: OK				  */
}


void * ModifyThreadData ( struct lt_funcArg * funcArgPtr )

/* Zhlerstnde verndern und anzeigen
 *
 * Argument:
 *   funcArgPtr	in/out	Thread-Namen und Zeiger auf eine Variable der
 *			aufrufenden Funktion
 *
 * Globale Variable:
 *   gvCallCnt	in/out	globale Zhlervariable
 *
 *
 * Diese Funktion verwendet eine lokale Variable und greift auerdem auf
 * Variablen der aufrufenden Funktion sowie auf die globale Variable zu.
 *
 * Die Funktion ruft zunchst PrintThreadStatus() auf, um die Zhleradressen
 * und -inhalte auszugeben. Anschlieend werden die Zhlerinhalte verndert.
 * Nach einer kurzen Wartezeit werden die Zhlerdaten nochmals angezeigt.
 */
{
  int
    localCounter;	     /* lokale Variable				  */

	  /* Inhalt des Zhlers der aufrufenden Funktion in lokale
	   * Variable bernehmen					  */
  localCounter = * funcArgPtr->mainCntPtr;
  
	  /* Label A: Threadnamen, Zhleradressen und -stnde anzeigen.
	   *          Fr die Zhler werden jeweils Zeiger auf die Zhler
	   *          bergeben.					  */
  PrintThreadStatus ( 'A', funcArgPtr->threadName, 
		      funcArgPtr->mainCntPtr,	& localCounter );

	  /* Inhalte aller Zhler ndern				  */
  gvCallCnt ++ ;
  * funcArgPtr->mainCntPtr +=  2;
  localCounter             += 15;
  
	  /* Label B: Threadnamen, Zhleradressen und -stnde nochmals
	   *          anzeigen						  */
  PrintThreadStatus ( 'B', funcArgPtr->threadName,
		      funcArgPtr->mainCntPtr,	& localCounter );

	  /* kurze Zeit warten						  */
  sleep ( 1 );
  
	  /* Label C: Threadnamen, Zhleradressen und -stnde nochmals
	   *          anzeigen						  */
  PrintThreadStatus ( 'C', funcArgPtr->threadName,
		      funcArgPtr->mainCntPtr,	& localCounter );

  return NULL;
}


void PrintThreadStatus ( char label, char * threadName, 
			 int * mainCntPtr, int * localCntPtr )

/* Zhleradressen und -inhalte anzeigen 
 *
 * Argument:
 *   label	  in	Markierung der Stelle, von der aus die Funktion
 *			aufgerufen wurde
 *   threadName	  in	Bezeichnung des Threads
 *   mainCntPtr	  in	Zeiger auf mainCounter  in main ()
 *   localCntPtr  in	Zeiger auf localCounter in ModifyThreadData()
 *
 * Globale Variable:
 *   gvCallCnt	  in	globale Zhlervariable
 *
 * Die Funktion schreibt die Inhalte und Adressen der bergebenen Zeiger
 * und der globalen Variablen auf den Bildschirm.
 * Die Funktion gibt die Daten ohne erluternden Text aus. Es wird voraus-
 * gesetzt, da vor dem ersten Aufruf der Funktion passende Spaltenber-
 * schriften ausgegeben wurden.
 */
{
	  /* Label, Threadbezeichnung, Zhleradressen und -stnde
	   * anzeigen							  */
  printf ( "%c  %-9s    * %p = %2d    * %p = %2d    ",
	   label, threadName, & gvCallCnt, gvCallCnt,
			      mainCntPtr,  * mainCntPtr );

	  /* Kein NULL-Pointer fr den lokalen Zhler ?			  */
  if ( NULL != localCntPtr )
  {
    printf ( "* %p = %2d\n", localCntPtr, * localCntPtr );
  }
  else
  {
    printf ( "  (NULL) \n" );
  }
}
