/**************************************************************************
 *
 * Beispiel zu Semaphoren
 *                                                                        
 * Ersteller:	Ingo Phleps                                          
 *                                                                        
 * Stand:	25.03 2003
 *
 * Aufruf:	shm_demo Sem-Key
 *
 *		  Sem-Key:   numerischer Schlssel als Key zur Identifi-
 *			     kation des zu verwendenden Semaphor-Satzes.
 *                                                                        
 * Zielsystem:	UNIX                                                 
 *                                                                        
 * bersetzen:	cc sem_demo.c -o sem_demo
 *
 * Beispiel zur Vorlesung "Rechnerorganisation"     5. und 6. Halbjahr
 * Fachrichtung Nachrichtentechnik an der Berufsakademie Stuttgart
 *
 * Funktionen des Moduls:
 *
 *   main		()   Semaphor anlegen und verwalten
 *   ReadArg		()   Key fr Semaphor-Satz aus Befehlszeile lesen
 *   SemDemo		()   Befehle lesen und Semaphor belegen bzw.
 *			     freigeben
 *   ReadCommand	()   Befehl fr Semaphor-Aktion einlesen
 *   PrintCommands	()   Gltige Befehle anzeigen
 *   PrintSemStatus	()   Aktuellen Status des Semaphors anzeigen
 *                                                                        
 **************************************************************************/

	/* Semaphore sind nicht Teil des POSIX-Standards, sondern im XPG-
	 * Standard definiert!						  */
# define _XOPEN_SOURCE

# include <stdio.h>
# include <time.h>
# include <unistd.h>
# include <stdlib.h>
# include <string.h>
# include <errno.h>
# include <assert.h>
# include <sys/ipc.h>
# include <sys/sem.h>


	/* --- Konstanten ----------------------------------------------- */

# ifndef TRUE
#  define TRUE		1
#  define FALSE		0
# endif

			     /* Kennzeichen fr ungltige Semaphor-ID	  */
# define SEM_ID_INVALID	-1

			     /* Zahl der Semaphore des Semaphor-Satzes	  */
# define SEM_ITEMS	 1

			     /* Index des Semaphors, mit dem gearbeitet
			      * wird					  */
# define DEMO_SEM_INDEX	 0


/*  */
	/* --- Typdeklarationen ----------------------------------------- */



	/* --- globale Variablen ---------------------------------------- */

static char                  /* RCS-Versionsnummer ...                    */
  * header = "$Id: //depot/rech_org/skript/src/sem_demo.c#1 $";



	/* --- Prototypen ----------------------------------------------- */

	/* Hauptprogram: Semaphor anlegen und verwalten			  */
int main ( int argc, char * argv[] );

	/* Key fr Semaphor-Satz aus Befehlszeile lesen			  */
int ReadArg ( int argc, char * argv[], key_t * semKey );

	/* Befehle lesen und Semaphor belegen bzw. freigeben		  */
void SemDemo ( int semId );

	/* Befehl fr Semaphor-Aktion einlesen				  */
int ReadCommand     ( void );

	/* Gltige Befehle anzeigen					  */
void PrintCommands  ( void );

	/* Aktuellen Status des Semaphors anzeigen			  */
void PrintSemStatus ( int semId );


	/* --- Funktionen ----------------------------------------------- */

int main ( int argc, char * argv[] )

/* Hauptprogramm: Semaphor anlegen und verwalten
 *
 * Argumente:
 *   argc	in	Zahl der Elemente in argv[]
 *   argv	in	Elemente der Kommandozeile des Programmaufrufs:
 *			  argv [0]:  Programmname
 *			  argv [1]:  Key zur Auswahl des Semaphor-Satzes
 *
 * Ergebnis:
 *   EXIT_SUCCESS	Programm ohne Fehler beendet
 *
 *   EXIT_FAILURE	Programm wegen Fehler abgebrochen
 *
 * Das Programm verbindet sich mit dem Semaphor-Satz, der zu dem als
 * Argument bergebenen Key gehrt. Wenn der Semaphor-Satz noch nicht
 * existiert, wird er automatisch angelegt und initialisiert. 
 * Danach wird die Funktion aufgerufen, die das Setzen und Freigeben des
 * Semaphors ermglicht. 
 *
 * Wenn das Semaphor von diesem Proze angelegt wurde, wird es vor dem
 * regulren Ende des Programms wieder gelscht. 
 */
{
  key_t
    semKey;		     /* Schlssel zur Auswahl des Semaphor-Satzes */

  int
    semId,		     /* ID zur Identifikation des Semaphor-Satzes */
    fRemoveSem = FALSE,	     /* TRUE: Semaphor-Satz vor Ende des Programms
			      *       lschen				  */
			     /* Exit-Status dieses Programms		  */
    exitStatus = EXIT_SUCCESS;

  char
    * progName;		     /* Name dieses Programms			  */

  
	  /* Name des Programms aus der Befehlszeile festhalten		  */
  progName = argv[0];


	  /* Schlssel zur Auswahl des Semaphor-Satzes aus Argument des
	   * Programmaufrufs lesen:
	   * Fehler ?							  */
  if ( EXIT_SUCCESS != ReadArg ( argc, argv, & semKey ) )
  {
	    /* Programm wegen Fehler beenden. Die Fehlermeldung wurde
	     * schon von ReadArg() ausgegeben.				  */
    exit ( EXIT_FAILURE );
  }

	  /* Semaphor-Satz anfordern, der zum Key gehrt.
	   * Um herauszubekommen, ob der Satz jetzt neu angelegt wird und
	   * deshalb von diesem Proze initialisiert werden mu, erfolgt
	   * der erste Versuch mit den Flags O_CREAT | O_EXCL.		  */
  semId = semget ( semKey, SEM_ITEMS, 0644 | IPC_CREAT | IPC_EXCL );

  	  /* Wurde ein neuer Semaphor-Satz angelegt?			  */
  if ( -1 != semId )
  {
	    /* Semaphor-Satz vor dem Beenden des Programms wieder lschen */
    fRemoveSem = TRUE;
    
	    /* Semaphor mit Wert 1 initialisieren.
	     * Fehler?							  */
    if ( -1 == semctl ( semId, DEMO_SEM_INDEX, SETVAL, /* semun.val = */ 1 ) )
    {
	      /* Fehlermeldung, Abbruch					  */
      perror ( "semctl ( SETVAL )" );

      exit ( EXIT_FAILURE );
    }

    printf ( "PID %d: Semaphor-Satz wurde neu angelegt und initialisiert \n\n",
	                                                     (int) getpid() );
  }
	    /* Fehler beim Anlegen eines neuen Semaphor-Satzes, obwohl es
	     * den Semaphor-Satz noch nicht gibt?			  */
  else if ( EEXIST != errno )
  {
	    /* Fehlermeldung, Abbruch					  */
    perror ( "semget ( IPC_CREAT | O_EXCL )" );

    exit ( EXIT_FAILURE );
  }
	  /* Semaphor-Satz existiert schon:
	   * Vorhandenen Semaphor-Satz anfordern.
	   * Fehler ?							  */
  else if ( -1 == ( semId = semget ( semKey, SEM_ITEMS, 0644 ) ) )
  {
	    /* Fehlermeldung, Abbruch					  */
    perror ( "semget ()" );

    exit ( EXIT_FAILURE );
  }

  
	  /* Aktionen mit Semaphoren ausfhren				  */
  SemDemo ( semId );

	  /* Soll der Semaphor-Satz von diesem Proze gelscht werden?	  */
  if ( TRUE == fRemoveSem )
  {
	  /* Semaphor-Satz wieder lschen, sobald kein Proze mehr damit
	   * verbunden ist
	   * Fehler?							  */
    if ( -1 == semctl ( semId, 0, IPC_RMID,0 ) )
    {
	      /* Fehlermeldung, Exit-Status: Fehler			  */
      perror ( "semctl ( IPC_RMID )" );
      
      exitStatus = EXIT_FAILURE;
    }

    printf ( "\nPID %d: Semaphor-Satz wurde gelscht \n", (int) getpid() );
  }
  printf ( "\n%s wird beendet. \n\n", progName );

  return exitStatus;

} /*  Ende  main ()  */


int ReadArg ( int argc, char * argv[], key_t * semKey )

/* Schlssel zur Auswahl des Semaphor-Satzes aus Befehlszeile lesen
 *
 * Argumente:
 *   argc	in	Zahl der Elemente in argv[]
 *   argv	in	Elemente der Kommandozeile des Programmaufrufs:
 *			  argv [0]:  Programmname
 *			  argv [1]:  Key zur Auswahl des Shared Memories
 *   semKey	out	Zeiger auf Ziel fr Schlssel zur Auswahl des
 *			Semaphor-Satzes
 *
 * Ergebnis:
 *   EXIT_SUCCESS	Kein Fehler: Schlssel fr Auswahl des Semaphor-
 *			     Satzes in *semKey eingetragen
 *
 *   EXIT_FAILURE	Fehler: Schlssel fr Auswahl des Semaphor-Satzes
 *			     konnte nicht gelesen werden.
 *			     Fehlermeldung wurde ausgegeben und *semKey = -1
 *			     gesetzt.
 *
 * Die Funktion liest den Key zur Auswahl des Semaphor-Satzes aus der
 * Befehlszeile des Programmaufrufs und schreibt den Schlssel nach
 * *semKey.
 *
 * Wenn dies wegen eines Fehlers nicht mglich war, gibt die Funktion eine
 * Fehlermeldung aus.
 */
{
  char
    * progName;		     /* Name dieses Programms			  */

	  /* Name des Programms aus der Befehlszeile festhalten		  */
  progName = argv[0];

	  /* Zielvariable fr Semaphor-ID mit Wert fr 'ungltig'
	   * vorbesetzen						  */
  * semKey = -1;

	  /* Aufruf ohne Argument?					  */
  if ( argc < 2 )
  {
	    /* Fehlermeldung, Ergebnis = "Fehler"			  */
    fprintf ( stderr,
	      "\n%s: Fehler: Argument mit Zahl fr Semaphor-Key fehlt!\n",
	      progName );
    fprintf ( stderr,
	        "        Programm wird abgebrochen.\n" );
    fprintf ( stderr,
	      "\n        Aufruf: %s Semaphor-Key\n", progName );
    fprintf ( stderr,
	        "                mit Dezimalzahl fr Semaphor-Key \n\n" );

    return EXIT_FAILURE;     /* Ergebnis: Fehler			  */
  }
	  /* Semaphor-Key lesen, Format umwandeln und in Zielvariable
	   * eintragen.
	   * Kein numerisches Argument fr Semaphor-Key?		  */
  else if ( 1 != sscanf ( argv[1], "%d", semKey ) )
  {
	    /* Fehlermeldung, Ergebnis = "Fehler"			  */
    fprintf ( stderr,
	 "\n%s: Fehler: 1. Argument fr Semaphor-Key ist keine Dezimalzahl!\n",
	      progName );
    fprintf ( stderr,
	        "        Programm wird abgebrochen.\n" );
    fprintf ( stderr,
	      "\n        Aufruf: %s Semaphor-Key\n", progName );
    fprintf ( stderr,
	        "                mit Dezimalzahl fr Semaphor-Key \n\n" );

    return EXIT_FAILURE;     /* Ergebnis: Fehler			  */
  }

  return EXIT_SUCCESS;	     /* Ergebnis: Kein Fehler			  */

}  /*  Ende  ReadArg ()  */


void SemDemo ( int semId )

/* Befehle lesen und Semaphor entsprechend belegen bzw. freigeben
 *
 * Argument:
 *   semId	in	ID zur Identifikation des Semaphor-Satzes
 *
 * Ergebnis:
 *   keines
 *
 * Die Funktion liest Befehle fr Semaphor-Aktionen von der Tastatur ein
 * und fhrt die entsprechenden Aktionen fr das Semaphor aus.
 */
{
  char
    command;		     /* auszufhrender Befehl:
			      *   p  Semaphor belegen
			      *   v  Semaphor freigeben
			      *   s  Semaphor-Status anzeigen
			      *   q  Programm beenden			  */

  struct sembuf
    semBuf [ SEM_ITEMS ];    /* auf den Semaphor-Satz anzuwendende
			      * Operationen
			      * In diesem Beispiel besteht der Semaphor-
			      * Satz nur aus einem Semaphor. Um dies bei
			      * Bedarf leichter erweitern zu knnen, wird
			      * hier trotzdem ein Array verwendet.	  */
  
  
	  /* Befehl fr Semaphor-Aktion einlesen und ausfhren		  */
  while ( 'q' != ( command = ReadCommand () ) )
  {
	    /* Befehl auswerten						  */
    switch ( command )
    {
      case 'p':		     /* Semaphor belegen			  */
			     /*   Semaphor auswhlen			  */
	  semBuf[0].sem_num = DEMO_SEM_INDEX;
			     /*   Aktion: mit Anzahl 1 belegen		  */
	  semBuf[0].sem_op  = -1;
			     /*   Operation automat. rckgngig machen,
			      *   wenn der Proze endet			  */
	  semBuf[0].sem_flg = SEM_UNDO; 

		  /* Fehler beim Belegen des Semaphors?			  */
	  if ( 0 != semop ( semId, semBuf, 1 ) )
	  {
		    /* Fehlermeldung ausgeben				  */
	    perror ( "semop ( P )" );
	  }
	  else
	  {
	    printf ( "PID %d: Semaphor wurde belegt \n", (int) getpid() );
	  }

	  break;

      case 'v':		     /* Semaphor freigeben:			  */
			     /*   Semaphor auswhlen			  */
	  semBuf[0].sem_num = DEMO_SEM_INDEX;
			     /*   Aktion: mit Anzahl 1 freigeben	  */
	  semBuf[0].sem_op  = 1;
			     /*   Operation automat. rckgngig machen,
			      *   wenn der Proze endet			  */
	  semBuf[0].sem_flg = SEM_UNDO; 

		  /* Fehler beim Freigeben des Semaphors?		  */
	  if ( 0 != semop ( semId, semBuf, 1 ) )
	  {
		    /* Fehlermeldung ausgeben				  */
	    perror ( "semop ( V )" );
	  }
	  else
	  {
	    printf ( "PID %d: Semaphor wurde freigegeben \n", (int) getpid() );
	  }

	  break;

      case 's':		     /* Semaphor-Status anzeigen		  */
	  PrintSemStatus ( semId );

	  break;

      default:		     /* darf nie vorkommen, da ReadCommand() auf
			      * zulssige Eingabe prft.		  */
	  assert ( FALSE );
	  break;

    }  /*  Ende switch ( command )  */

  }  /*  Ende while ( command )  */
	  
}  /*  Ende  SemDemo ()  */


int ReadCommand ( void )

/* Befehl fr Semaphor-Aktion einlesen 
 *
 * Argument:
 *   keines
 *
 * Ergebnis:
 *   eingegebener Befehl:
 *		p  Semaphor belegen
 *		v  Semaphor freigeben
 *		s  Semaphor-Status anzeigen
 *		q  Programm beenden
 *
 * Die Funktion schreibt eine Aufforderung zur Eingabe eines Befehls auf
 * den Bildschirm und liest den Befehl von der Tastatur ein.
 *
 * Danach wird die Eingabe geprft. Bei ungltigen Eingaben wird eine
 * Fehlermeldung ausgegeben und das Einlesen des Befehls wiederholt.
 */
{
  char
    command,		     /* eingegebener Befehl			  */
    inLine [ 20 ];	     /* Puffer fr Eingabe			  */

  int
    fRepeat = TRUE;	     /* TRUE: wegen Fehler: Neuen Befehl lesen	  */

  do
  {
	    /* Eingabe-Aufforderung ausgeben				  */
    printf ( "Bitte Befehl eingeben ( p / v / s / q / ? ): " );
	    /* Ausgabe trotz line-buffering erzwingen			  */
    fflush ( stdout );

	    /* Eingegebenen Befehl lesen.
	     * Fehler ?							  */
    if ( NULL == fgets ( inLine, sizeof ( inLine ), stdin ) )
    {
	      /* Fehlermeldung, Programmabbruch				  */
      perror ( "fgets ()" );
      exit ( EXIT_FAILURE );
    }

    /* Befehl prfen						  */
    switch ( command = inLine [0] )
    {
      case 'p':
      case 'v':
      case 's':
      case 'q':		     /* gltigen Befehl akzeptieren		  */
	  fRepeat = FALSE;
	  break;

      case '?':		     /* gltige Befehle anzeigen		  */
	  PrintCommands ();
			     /* weiteren Befehl lesen			  */
	  fRepeat = TRUE;

	  break;

      default:		     /* ungltiger Befehl:
			      * Fehlermeldung ausgeben,			  */
	  printf ( "\nFehler: Eingegebener Befehl '%c' ist ungltig!\n",
								 command );
	  
			     /* gltige Befehle anzeigen		  */
	  PrintCommands ();
			     /* weiteren Befehl lesen			  */
	  fRepeat = TRUE;

	  break;

    } /* Ende switch  */

  } while ( TRUE == fRepeat );

  return ( command );	     /* Ergebnis: eingegebener Befehl		  */

}  /*  Ende  ReadCommand ()  */



void PrintCommands ( void )

/* Gltige Befehle anzeigen
 *
 * Argument:
 *   keines
 *
 * Ergebnis:
 *   keines
 *
 * Die Funktion schreibt die gltigen Befehle und iher Erklrung auf den
 * Bildschirm.
 */
{

  printf ( "\nGltige Befehle:        p  Semaphor belegen \n" );
  printf (   "                        v  Semaphor freigeben \n" );
  printf (   "                        s  Semaphor-Status anzeigen \n" );
  printf (   "                        q  Programm beenden \n\n" );

}  /*  Ende  PrintCommands ()  */


void PrintSemStatus ( int semId )

/* Aktuellen Status des Semaphors anzeigen
 *
 * Argument:
 *   semId	in	ID zur Identifikation des Semaphor-Satzes
 *
 * Ergebnis:
 *   keines
 *
 * Die Funktion liest Status-Informationen des Semaphors und schreibt sie
 * auf den Bildschirm.
 */
{
  int
    lastPid,		     /* Proze-ID des Prozesses, der das Semaphor
			      * zuletzt manipuliert hat			  */
    waitingProc;	     /* Zahl der Prozesse, die auf Freigabe des
			      * Semaphors warten			  */


	  /* ID des Prozesse mit der letzten Semaphor-Aktion lesen	  */
  lastPid = semctl ( semId, DEMO_SEM_INDEX, GETPID,
		       /* Dummy fr semun.val = */ 0 );
  
	  /* Fehler ?							  */
  if ( -1 == lastPid )
  {
	    /* Fehlermeldung						  */
    perror ( "semctl ( GETPID )" );
  }
  else
  {
	    /* ID des Prozesse mit der letzten Semaphor-Aktion anzeigen	  */
    printf ( "Letzte Semaphor-Aktion durch Proze mit PID       %5d\n",
	     lastPid );
  }

  
	  /* aktuellen Wert des Semaphors lesen				  */
  waitingProc = semctl ( semId, DEMO_SEM_INDEX, GETVAL,
			   /* Dummy fr semun.val = */ 0 );

	  /* Fehler ?							  */
  if ( -1 == waitingProc )
  {
	    /* Fehlermeldung						  */
    perror ( "semctl ( GETVAL )" );
  }
  else
  {
    printf ( "Aktueller Wert des Semaphors:                         %d\n",
	     waitingProc );
  }
	  
	  /* Zahl der Prozesse lesen, die warten, bis sie das Semaphor
	   * belegen knnen						  */
  waitingProc = semctl ( semId, DEMO_SEM_INDEX, GETNCNT,
			    /* Dummy fr semun.val = */ 0 );

	  /* Fehler ?							  */
  if ( -1 == waitingProc )
  {
	    /* Fehlermeldung						  */
    perror ( "semctl ( GETNCNT )" );
  }
  else
  {
    printf ( "Anzahl der Prozesse, die das Semaphor belegen wollen: %d\n",
	     waitingProc );
  }
	  
}  /*  Ende  PrintSemStatus ()  */




