/**************************************************************************
 *
 * Beispiel zu Shared Memory: 
 * Teil 2: Daten aus Shared Memory lesen und zum Schlu Shared Memory lschen
 *                                                                        
 * Ersteller:	Ingo Phleps                                          
 *                                                                        
 * Stand:	23.03 2003
 *
 * Aufruf:	shm_recv SHM-Key
 *
 *		  SHM-Key:   numerischer Schlssel als Key zur Identifikation
 *			     des Shared Memories.
 *			     shm_send und shm_recv mssen mit dem selben
 *			     Wert fr SHM-Key aufgerufen werden.
 *                                                                        
 * Zielsystem:	UNIX                                                 
 *                                                                        
 * bersetzen:	cc shm_recv.c shm_common.c -o shm_recv
 *
 * Beispiel zur Vorlesung "Rechnerorganisation"     5. und 6. Halbjahr
 * Fachrichtung Nachrichtentechnik an der Berufsakademie Stuttgart
 *
 * Dieses Beispiel zu Shared Memories besteht aus zwei Programmen:
 * - shm_send	legt das Shred Momory an, sofern es noch nicht existiert,
 *		und schreibt Daten in den Bereich.
 * - shm_recv	liest die Daten aus dem Shared Memory und lscht danach
 *		das Shared Memory.
 *
 * Das Beispiel verwendet absichtlich keinen Mechanismus zum Synchronisieren
 * von Lese- und Schreibzugriffen um die Folgen von Zugriffen bei fehlender
 * Synchronisierung zeigen zu knnen. 
 *
 * Die Datenstruktur fr die im Shared Memory abgelegten Daten ist in
 * shm_demo.h definiert.
 *									  
 * Funktionen des Moduls:
 *
 *   main		()   Shared Memory verwalten
 *   ShmRecv		()   Nachrichten aus Shared Memory lesen und
 *			     anzeigen
 *                                                                        
 **************************************************************************/

	/* Shared Memories sind nicht Teil des POSIX-Standards, sondern
	 * im XPG-Standard definiert!					  */
# define _XOPEN_SOURCE

# include <stdio.h>
# include <stdlib.h>
# include <string.h>
# include <sys/ipc.h>
# include <sys/shm.h>

# include "shm_common.h"


	/* --- globale Variablen ---------------------------------------- */

static char                  /* RCS-Versionsnummer ...                    */
  * header = "$Id: //depot/rech_org/skript/src/shm_recv.c#2 $";


/*  */
	/* --- Prototypen ----------------------------------------------- */

	/* Hauptprogram: Shared Memory verwalten			  */
int main ( int argc, char * argv[] );

	/* Nachrichten aus Shared Memory lesen und anzeigen		  */
int ShmRecv ( char * progName, struct gt_shmData * shmData );



	/* --- Funktionen ----------------------------------------------- */

int main ( int argc, char * argv[] )

/* Hauptprogramm: Shared Memory verwalten
 *
 * Argumente:
 *   argc	in	Zahl der Elemente in argv[]
 *   argv	in	Elemente der Kommandozeile des Programmaufrufs:
 *			  argv [0]:  Programmname
 *			  argv [1]:  Key zur Auswahl des Shared Memories
 *
 * Ergebnis:
 *   EXIT_SUCCESS	Programm ohne Fehler beendet
 *
 *   EXIT_FAILURE	Programm wegen Fehler abgebrochen
 *
 * Das Programm verbindet sich mit dem Shared Memory. Das Shared Memory mu
 * dazu bereits vom Program shm_send angelegt worden sein.
 * Anschlieend liest das Programm Nachrichten aus dem Shared Memory.
 * Vor dem Ende des Programms wird das Shared Memory gelscht.
 */
{
  key_t
    shmKey;		     /* Schlssel zur Auswahl des Shared Memories */

  int
    shmId,		     /* ID zur Identifikation des Shared Memories */
    msgCount,		     /* Zahl der empfangenen Meldungen		  */
			     /* Exit-Status dieses Programms		  */
    exitStatus = EXIT_SUCCESS;

  struct gt_shmData
    * shmData;		     /* Zeiger auf Inhalt des Shared Memories	  */

  char
    * progName;		     /* Name dieses Programms			  */

  
	  /* Name des Programms aus der Befehlszeile festhalten		  */
  progName = argv[0];


	  /* Schlssel zur Auswahl des Shared Memories aus Argument des
	   * Programmaufrufs lesen:
	   * Fehler ?							  */
  if ( EXIT_SUCCESS != ReadArg ( argc, argv, & shmKey ) )
  {
	    /* Programm wegen Fehler beenden. Die Fehlermeldung wurde
	     * schon von ReadArg() ausgegeben.				  */
    exit ( EXIT_FAILURE );
  }

/*  */
	  /* ID des Shared Memories aus Schlssel ermitteln.
	   * Das Shared Memory mu schon vorhanden sein.		  */
  shmId = shmget ( shmKey, sizeof ( struct gt_shmData ), 0644 );
  
	  /* Fehler beim Ermitteln der ID?				  */
  if ( -1 == shmId )
  {
	    /* Fehlermeldung, Abbruch					  */
    perror ( "shmget ()" );

    exit ( EXIT_FAILURE );
  }

	  /* Proze mit Shared Memory verbinden, nur Lesezugriff
	   * notwendig							  */
  shmData = shmat ( shmId, NULL, SHM_RDONLY );

	  /* Fehler beim Verbinden ?					  */
  if ( NULL == shmData )
  {
	    /* Fehlermeldung, Abbruch					  */
    perror ( "shmat ()" );

    exit ( EXIT_FAILURE );
  }

	  /* Daten aus dem Shared Memory lesen				  */
  msgCount = ShmRecv ( progName, shmData );

	  /* Proze vom Shared Memory trennen.
	   * Fehler?							  */
  if ( -1 == shmdt ( shmData ) )
  {
	    /* Fehlermeldung, Exit-Status: Fehler			  */
    perror ( "shmdt ()" );

    exitStatus = EXIT_FAILURE;
  }

	  /* Shared Memory lschen, sobald kein Proze mehr damit
	   * verbunden ist
	   * Fehler?							  */
  if ( -1 == shmctl ( shmId, IPC_RMID, NULL ) )
  {
	    /* Fehlermeldung, Exit-Status: Fehler			  */
    perror ( "shmctl ()" );

    exitStatus = EXIT_FAILURE;
  }

  printf ( "\n%s wird beendet: %d Nachricht(en) empfangen. \n\n",
	   progName, msgCount );

  return exitStatus;

} /*  Ende  main ()  */


/*  */
int ShmRecv ( char * progName, struct gt_shmData * shmData )

/* Nachrichten aus Shared Memory lesen und anzeigen
 *
 * Argumente:
 *   progName	in	Name dieses Programms
 *   shmData	in	Zeiger auf Inhalt des Shared Memories
 *
 * Ergebnis:
 *   Zahl der empfangenen Meldungen
 *
 * Die Funktion liest die Daten aus dem Shared Memory und zeig sie an.
 * Anschlieend wird jeweils auf eine Eingabe des Anwenders gewartet.
 */
{
  int
    msgCount = 0;	     /* Zahl des geschriebenen Meldungen	  */

	  /* Nachricht lesen und anzeigen				  */
  do
  {

	    /* Inhalt des Shared memories anzeigen
	     * Auf eine normalerweise notwendige Synchronisierung der
	     * Lese- und Schreibzugriffe wird hier absichtlich verzichtet,
	     * um die Folgen von Zugriffen bei fehlender Synchronisierung
	     * zeigen zu knnen.					  */
    printf ( "Inhalt des Shared Memory: PID %d, Text: \"%s\"\n",
	     (int) shmData->sendPid, shmData->shmText );

	    /* Zahl der empfangenen Nachrichten weiterzhlen		  */
    msgCount ++ ;

	    /* auf Eingabe warten und je nach Eingabe Inhalt des Shared
	     * Memories nochmals lesen, oder lesen beenden.		  */
  } while ( TRUE == AskForMore ( "lesen" ) );


	  /* Ergebnis: Zahl der gelesenen Nachrichten			  */
  return msgCount;

}  /*  Ende  ShmRecv ()  */
