/**************************************************************************
 *
 * Beispiel zu Shared Memory: 
 * Teil 1: Shared Memory anlegen und Daten dort ablegen
 *                                                                        
 * Ersteller:	Ingo Phleps                                          
 *                                                                        
 * Stand:	23.03 2003
 *
 * Aufruf:	shm_send SHM-Key
 *
 *		  SHM-Key:   numerischer Schlssel als Key zur Identifikation
 *			     des Shared Memories.
 *			     shm_send und shm_recv mssen mit dem selben
 *			     Wert fr SHM-Key aufgerufen werden.
 *                                                                        
 * Zielsystem:	UNIX                                                 
 *                                                                        
 * bersetzen:	cc shm_send.c shm_common.c -o shm_send
 *
 * Beispiel zur Vorlesung "Rechnerorganisation"     5. und 6. Halbjahr
 * Fachrichtung Nachrichtentechnik an der Berufsakademie Stuttgart
 *
 * Dieses Beispiel zu Shared Memories besteht aus zwei Programmen:
 * - shm_send	legt das Shred Momory an, sofern es noch nicht existiert,
 *		und schreibt Daten in den Bereich.
 * - shm_recv	liest die Daten aus dem Shared Memory und lscht danach
 *		das Shared Memory.
 *
 * Das Beispiel verwendet absichtlich keinen Mechanismus zum Synchronisieren
 * von Lese- und Schreibzugriffen um die Folgen von Zugriffen bei fehlender
 * Synchronisierung zeigen zu knnen. 
 *
 * Die Datenstruktur fr die im Shared Memory abgelegten Daten ist in
 * shm_demo.h definiert.
 *									  
 * Funktionen des Moduls:
 *
 *   main		()   Shared Memory anlegen  
 *   ShmSend		()   Nachrichten in Shared Memory schreiben
 *                                                                        
 **************************************************************************/

	/* Shared Memories sind nicht Teil des POSIX-Standards, sondern
	 * im XPG-Standard definiert!					  */
# define _XOPEN_SOURCE

# include <stdio.h>
# include <time.h>
# include <unistd.h>
# include <stdlib.h>
# include <string.h>
# include <sys/ipc.h>
# include <sys/shm.h>

# include "shm_common.h"


	/* --- globale Variablen ---------------------------------------- */

static char                  /* RCS-Versionsnummer ...                    */
  * header = "$Id: //depot/rech_org/skript/src/shm_send.c#2 $";


/*  */
	/* --- Prototypen ----------------------------------------------- */

	/* Hauptprogram: Shared Memory anlegen				  */
int main ( int argc, char * argv[] );

	/* Nachrichten in Shared Memory schreiben			  */
int ShmSend ( char * progName, struct gt_shmData * shmData );



	/* --- Funktionen ----------------------------------------------- */

int main ( int argc, char * argv[] )

/* Hauptprogramm: Shared Memory anlegen
 *
 * Argumente:
 *   argc	in	Zahl der Elemente in argv[]
 *   argv	in	Elemente der Kommandozeile des Programmaufrufs:
 *			  argv [0]:  Programmname
 *			  argv [1]:  Key zur Auswahl des Shared Memories
 *
 * Ergebnis:
 *   EXIT_SUCCESS	Programm ohne Fehler beendet
 *
 *   EXIT_FAILURE	Programm wegen Fehler abgebrochen
 *
 * Das Programm legt ein Shared Memory fr den als Argument bergebenen Key
 * an und schreibt Nachrichten in das Shared Memory.
 * Danach beendet sich das Programm. 
 */
{
  key_t
    shmKey;		     /* Schlssel zur Auswahl des Shared Memories */

  int
    shmId,		     /* ID zur Identifikation des Shared Memories */
    msgCount,		     /* Zahl der empfangenen Meldungen		  */
			     /* Exit-Status dieses Programms		  */
    exitStatus = EXIT_SUCCESS;

  struct gt_shmData
    * shmData;		     /* Zeiger auf Inhalt des Shared Memories	  */

  char
    * progName;		     /* Name dieses Programms			  */

  
	  /* Name des Programms aus der Befehlszeile festhalten		  */
  progName = argv[0];


	  /* Schlssel zur Auswahl des Shared Memories aus Argument des
	   * Programmaufrufs lesen:
	   * Fehler ?							  */
  if ( EXIT_SUCCESS != ReadArg ( argc, argv, & shmKey ) )
  {
	    /* Programm wegen Fehler beenden. Die Fehlermeldung wurde
	     * schon von ReadArg() ausgegeben.				  */
    exit ( EXIT_FAILURE );
  }

/*  */
	  /* Shared Memory mit Zugriffsrechten rw-r--r-- anlegen, wenn es
	   * nicht schon existiert, und ID des Shared Memories aus
	   * Schlssel ermitteln.					  */
  shmId = shmget ( shmKey, sizeof ( struct gt_shmData ), 0644 | IPC_CREAT );
  
	  /* Fehler beim Anlegen?					  */
  if ( -1 == shmId )
  {
	    /* Fehlermeldung, Abbruch					  */
    perror ( "shmget ()" );

    exit ( EXIT_FAILURE );
  }

	  /* Proze mit Shared Memory verbinden				  */
  shmData = shmat ( shmId, NULL, 0 );

	  /* Fehler beim Verbinden ?					  */
  if ( NULL == shmData )
  {
	    /* Fehlermeldung, Abbruch					  */
    perror ( "shmat ()" );

    exit ( EXIT_FAILURE );
  }

	  /* Daten in das Shared Memory schreiben			  */
  msgCount = ShmSend ( progName, shmData );

	  /* Proze vom Shared Memory trennen.
	   * Fehler?							  */
  if ( -1 == shmdt ( shmData ) )
  {
	    /* Fehlermeldung, Exit-Status: Fehler			  */
    perror ( "shmdt ()" );

    exitStatus = EXIT_FAILURE;
  }

  printf ( "\n%s wird beendet: %d Nachricht(en) gesendet. \n\n",
	   progName, msgCount );

  return exitStatus;

} /*  Ende  main ()  */


int ShmSend ( char * progName, struct gt_shmData * shmData )

/* Nachrichten in Shared Memory schreiben
 *
 * Argumente:
 *   progName	in	Name dieses Programms
 *   shmData	in	Zeiger auf Inhalt des Shared Memories
 *
 * Ergebnis:
 *   Zahl der geschriebenen Meldungen
 *
 * Die Funktion schreibt kurze Meldungen mit der Proze-ID und der
 * aktuellen Uhrzeit in das Shared Memory.
 * Anschlieend wird jeweils auf eine Eingabe des Anwenders gewartet.
 */
/*  */
{
  time_t
    now;		     /* aktuelle Uhrzeit			  */
  
  int
    msgCount = 0;	     /* Zahl des geschriebenen Meldungen	  */

  char			     /* Zwischenpuffer fr zu sendenden Text	  */
    text [ MAX_TEXT_LEN +1 ];

	  /* Nachricht zusammenbauen und senden				  */
  do
  {
	    /* aktuelle Uhrzeit ermitteln				  */
    now = time ( NULL );
    
	    /* Nachricht mit der Proze-ID und der aktuellen Uhrzeit
	     * erstellen.
	     * Die Nachricht knnte direkt im das Shared Memory ge-
	     * schrieben werden. In diesem Beispiel wird sie zunchst
	     * lokal zusammengebaut, damit der gesendete Text zur
	     * Kontrolle angezeigt werden kann.				  */
    strftime ( text, sizeof ( text ),
	       "gesendet am %d.%m.%Y, %H:%M:%S Uhr",
	       localtime ( & now ) );

	    /* zu sendende Nachricht anzeigen				  */
    printf ( "Proze %d sendet: \"%s\"\n", (int) getpid(), text );

	    /* eigene Proze-ID und zu sendenden Text in das Shared
	     * Memory bertragen.
	     * Auf eine normalerweise notwendige Synchronisierung der
	     * Lese- und Schreibzugriffe wird hier absichtlich verzichtet,
	     * um die Folgen von Zugriffen bei fehlender Synchronisierung
	     * zeigen zu knnen.					  */
    shmData->sendPid = getpid ();

    strncpy ( shmData->shmText, text, sizeof ( shmData->shmText ) );

	    /* Wenn der Text die maximal erlaubte Lnge erreicht hat,
	     * schreibt strncpy() kein abschlieendes '\0'!
	     * Es wird deshalb hier sicherheitshalber explizit gesetzt.	  */
    shmData->shmText [ sizeof ( shmData->shmText ) -1 ] = '\0';

	    /* Zahl der gesendeten Nachrichten weiterzhlen		  */
    msgCount ++ ;

	    /* auf Eingabe warten und je nach Eingabe neue Nachricht
	     * senden, oder senden beenden.				  */
  } while ( TRUE == AskForMore ( "senden" ) );


	  /* Ergebnis: Zahl der gesendeten Nachrichten			  */
  return msgCount;

}  /*  Ende  ShmSend ()  */
