/**************************************************************************
 * $Header: /home/phleps/ba/rech_org/skript/src/RCS/sig_posix.c,v 1.2 2001/04/25 18:04:59 phleps Exp $
 *
 * Beispiel zur Signalbehandlung entsprechend POSIX.1 = IEEE 1003.1-1990
 *
 * Ersteller :       Ingo Phleps                                          
 *                                                                        
 * Stand :           $Date: 2001/04/25 18:04:59 $
 *                                                                        
 * Zielsystem:       Systeme nach POSIX-Standard, z. B. UNIX
 *                                                                        
 * bersetzen:       cc sig_posix.c -o sig_posix
 *
 * 
 * Dieses Programm ist ein Beispiel fr die Signalbehandlung von SIGTERM
 * mit den im POSIX-Standard definierten Funktionen:
 * Das Programm installiert eine eigene Behandlungsfunktion fr SIGTERM.
 * Bei jedem empfangenen SIGTERM wird ein Zhler inkrementiert. Der aktuelle
 * Zhlerstand wird jeweils vom Hauptprogramm angezeigt. Nach dem fnften
 * empfangenen SIGTERM wird das Programm beendet. 
 * 
 *
 * Funktionen:
 *   main		()   Signalbehandlung aktivieren und Stand des
 *			     Signalzhlers anzeigen
 *   SigTermHandler	()   Signalbehandlung von SIGTERM
 *
 **************************************************************************/

/* ---- Definitionen zum ein-/ausschalten bedingter Uebersetzung -------- */

# define _POSIX_SOURCE    1  /* Uebersetzen entsprechend POSIX Standard	  */


/* ---- allgemeine Include-Dateien -------------------------------------- */

# include <stdio.h>
# include <signal.h>
# include <stdlib.h>
# include <unistd.h>
# include <time.h>


/* ---- Symbolische Konstanten, Makros: # define ... -------------------- */

	/* Zahl der empfangenen SIGTERM Signale, nach denen das Programm
	 * sich beenden soll						  */
# define EXIT_SIGTERM_CNT    5


/* ---- Definitionen von globalen und static-Variablen: ----------------- */

# ifndef __LINT__
	/* Objekt-Datei und Programm mit RCS-Infostrings kennzeichnen	  */
volatile static char
  * rcsHeader = "@(#)$Header: /home/phleps/ba/rech_org/skript/src/RCS/sig_posix.c,v 1.2 2001/04/25 18:04:59 phleps Exp $" ;
# endif   /* __LINT__  */

static int
  sigTermCnt = 0;	     /* Zahl der empfangenen SIGTERM Signale	  */


/* ---- Funktionsprototypen von static-Funktionen: ---------------------- */

	/* Signalbehandlung aktivieren, Stand des Signalzhlers anzeigen  */
int main ( void );

	/* Signalbehandlung von SIGTERM					  */
void SigTermHandler ( int signo );


/* ---- Funktionen: ----------------------------------------------------- */

int main ( void )

/* Funktionsergebnis:
 *   EXIT_SUCCESS	immer
 *
 * Funktionsparameter:
 *   keine
 *
 * Globale Variablen:
 *   sigTermCnt	in	Zahl der empfangenen SIGTERM Signale
 *
 * Signalbehandlung aktivieren und Stand des Signalzhlers anzeigen
 *
 * Die Funktion installiert SigTermHandler() zur Signalbehandlung von
 * SIGTERM. Danach wird periodisch der Zhlerstand von sigTermCnt angezeigt.
 *
 * Wenn der Zhlerstand den Wert EXIT_SIGTERM_CNT erreicht, wird die
 * ursprngliche Signalbehandlung fr SIGTERM wiederhergestellt und das
 * Programm beendet.
 */
{
  struct sigaction
    oldTermAction,	     /* ursprngliche Behandlung von SIGTERM	  */
    newTermAction;	     /* neue Behandlung von SIGTERM		  */

  time_t
    now ;		     /* aktuelle Zeit				  */
  
  char
    timeStr [ 10 ];	     /* aktuelle Zeit				  */

  
	  /* Signalbehandlung fr SIGTERM durch SigTermHandler()
	   * vorbereiten						  */
  newTermAction.sa_handler = SigTermHandler;
  sigemptyset ( & newTermAction.sa_mask );
  newTermAction.sa_flags   = 0;

	  /* Signalbehandlung fr SIGTERM durch SigTermHandler()
	   * aktivieren, alte Signalbehandlung merken.			  */
  sigaction ( SIGTERM, & newTermAction, &oldTermAction );

  printf ( "\nPID %d: Behandlung von SIGTERM "
	                    "durch SigTermHandler() aktiviert \n",
	   getpid () );


	  /* Zhlerstand fr Anzahl der empfangenen SIGTERM-Signale
	   * im Raster von ca. 3 Sekunden anzeigen, bis der Zhlerstand
	   * den definierten Grenzwert erreicht hat.			  */
  do
  {
    sleep ( 3 );
	    /* aktuelle Zeit ermitteln und fr Anzeige aufbereiten	  */
    now = time ( NULL ); 
    strftime ( timeStr, sizeof ( timeStr ),
	       "%H:%M:%S", localtime ( & now ) );
    
    printf ( "%s: SIGTERM wurde %2d mal empfangen \n",
	                                timeStr, sigTermCnt );
    
  } while ( sigTermCnt < EXIT_SIGTERM_CNT );    
  

	  /* ursprngliche Signalbehandlung wiederherstellen		  */
  sigaction ( SIGTERM, &oldTermAction, NULL );

  printf ( "ursprngliche Signalbehandlung wiederhergestellt\n" );
  
	  /* zu Vorfhrzwecken kurz warten				  */
  sleep ( 5 );

  printf ( "\nPID %d: Programm endet \n\n", getpid () );

  return ( EXIT_SUCCESS );

} /* Ende  main ()  */
  

void SigTermHandler ( int signo )

/* Funktionsergebnis:
 *   keines
 *
 * Funktionsparameter:
 *   signo	in	Signal, das den Aufruf dieser Signalbehandlungs-
 *			funktion ausgelst hat.
 *			Der Parameter wird von dieser Funktion nicht
 *			ausgewertet.
 *
 * Globale Variablen:
 *   sigTermCnt	out	Zahl der empfangenen SIGTERM Signale
 *
 * Signalbehandlung von SIGTERM
 *
 * Diese Funktion erhht bei jedem Aufruf den Inhalt von sigTermCnt um 1.
 * Sie ist dafr gedacht, als Signalbehandlungsfunktion fr SIGTERM
 * verwendet zu werden.
 */
{
	  /* Zhler fr empfangene SIGTERM Signale weiterzhlen		  */
  sigTermCnt ++ ;

} /* Ende  SigTermHandler ()  */
